<?php
/**
 * OptiCore SaaS - Venta Model
 */

class Venta extends BaseModel
{
    protected string $table = 'ventas';

    // ── Alias con estructura ['data','pagination'] ─────────────
    public function listar(int $page = 1, int $perPage = ITEMS_PER_PAGE, array $filters = []): array
    {
        $result = $this->getAll($page, $filters);
        return [
            'data'       => $result['data'],
            'pagination' => $result,
        ];
    }

    public function getAll(int $page = 1, array $filters = []): array
    {
        $params = [$this->empresaScope()];
        $where  = 'v.empresa_id = ?';

        if (!empty($filters['sucursal_id'])) {
            $where .= ' AND v.sucursal_id = ?'; $params[] = $filters['sucursal_id'];
        }
        if (!empty($filters['estado'])) {
            $where .= ' AND v.estado = ?'; $params[] = $filters['estado'];
        }
        if (!empty($filters['desde'])) {
            $where .= ' AND DATE(v.fecha) >= ?'; $params[] = $filters['desde'];
        }
        if (!empty($filters['hasta'])) {
            $where .= ' AND DATE(v.fecha) <= ?'; $params[] = $filters['hasta'];
        }
        if (!empty($filters['search'])) {
            $s = '%' . $filters['search'] . '%';
            $where .= " AND (v.correlativo LIKE ? OR CONCAT(p.nombre,' ',p.apellido) LIKE ?)";
            $params[] = $s; $params[] = $s;
        }

        $sql = "SELECT v.*,
                       CONCAT(u.nombre,' ',u.apellido) as vendedor,
                       CONCAT(p.nombre,' ',p.apellido) as paciente_nombre,
                       s.nombre as sucursal_nombre
                FROM ventas v
                LEFT JOIN usuarios u ON v.usuario_id = u.id
                LEFT JOIN pacientes p ON v.paciente_id = p.id
                LEFT JOIN sucursales s ON v.sucursal_id = s.id
                WHERE $where
                ORDER BY v.fecha DESC, v.id DESC";

        return $this->db->paginate($sql, $params, $page);
    }

    public function getById(int $id): array|false
    {
        return $this->db->fetchOne(
            "SELECT v.*,
                    CONCAT(u.nombre,' ',u.apellido) as vendedor,
                    CONCAT(p.nombre,' ',p.apellido) as paciente_nombre,
                    p.rut as paciente_rut, p.telefono as paciente_telefono,
                    s.nombre as sucursal_nombre, s.direccion as sucursal_direccion,
                    s.telefono as sucursal_telefono,
                    e.nombre as empresa_nombre, e.logo as empresa_logo,
                    e.direccion as empresa_direccion, e.telefono as empresa_telefono,
                    e.rut as empresa_rut, e.moneda
             FROM ventas v
             LEFT JOIN usuarios u ON v.usuario_id = u.id
             LEFT JOIN pacientes p ON v.paciente_id = p.id
             LEFT JOIN sucursales s ON v.sucursal_id = s.id
             LEFT JOIN empresas e ON v.empresa_id = e.id
             WHERE v.id = ? AND v.empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }

    public function getItems(int $ventaId): array
    {
        return $this->db->fetchAll(
            "SELECT vi.*, p.sku, p.tipo
             FROM venta_items vi
             LEFT JOIN productos p ON vi.producto_id = p.id
             WHERE vi.venta_id = ?",
            [$ventaId]
        );
    }

    public function getPagos(int $ventaId): array
    {
        return $this->db->fetchAll(
            "SELECT vp.*, fp.nombre as forma_pago_nombre, fp.tipo as forma_pago_tipo
             FROM venta_pagos vp
             JOIN formas_pago fp ON vp.forma_pago_id = fp.id
             WHERE vp.venta_id = ?",
            [$ventaId]
        );
    }

    public function crear(array $venta, array $items, array $pagos): int
    {
        $db = $this->db;
        try {
            $db->beginTransaction();

            // Obtener y actualizar correlativo
            $sucursal = $db->fetchOne(
                "SELECT correlativo_ventas FROM sucursales WHERE id = ? FOR UPDATE",
                [$venta['sucursal_id']]
            );
            $correlativo = generarCorrelativo('V-' . str_pad($venta['sucursal_id'], 3, '0', STR_PAD_LEFT) . '-', $sucursal['correlativo_ventas']);
            $db->query(
                "UPDATE sucursales SET correlativo_ventas = correlativo_ventas + 1 WHERE id = ?",
                [$venta['sucursal_id']]
            );

            // Calcular totales
            $subtotal = 0;
            $descuento = 0;
            $impuesto = 0;
            foreach ($items as $item) {
                $subtotal  += $item['subtotal'];
                $descuento += $item['descuento'] ?? 0;
                $impuesto  += $item['impuesto'] ?? 0;
            }
            $total = $subtotal - $descuento + $impuesto;

            // Insertar venta
            $ventaId = $db->insert('ventas', [
                'empresa_id'    => $this->empresaScope(),
                'sucursal_id'   => $venta['sucursal_id'],
                'usuario_id'    => Auth::id(),
                'paciente_id'   => $venta['paciente_id'] ?: null,
                'correlativo'   => $correlativo,
                'fecha'         => date('Y-m-d H:i:s'),
                'subtotal'      => $subtotal,
                'descuento'     => $descuento,
                'impuesto'      => $impuesto,
                'total'         => $total,
                'estado'        => $venta['estado'] ?? 'confirmada',
                'observaciones' => $venta['observaciones'] ?? null,
            ]);

            // Insertar items
            foreach ($items as $item) {
                $db->insert('venta_items', [
                    'venta_id'        => $ventaId,
                    'producto_id'     => $item['producto_id'],
                    'nombre_producto' => $item['nombre_producto'],
                    'cantidad'        => $item['cantidad'],
                    'precio_unitario' => $item['precio_unitario'],
                    'descuento'       => $item['descuento'] ?? 0,
                    'impuesto'        => $item['impuesto'] ?? 0,
                    'subtotal'        => $item['subtotal'],
                ]);
            }

            // Insertar pagos
            foreach ($pagos as $pago) {
                if ($pago['monto'] > 0) {
                    $db->insert('venta_pagos', [
                        'venta_id'      => $ventaId,
                        'forma_pago_id' => $pago['forma_pago_id'],
                        'monto'         => $pago['monto'],
                        'referencia'    => $pago['referencia'] ?? null,
                    ]);
                }
            }

            // Descontar stock si es venta confirmada
            if (($venta['estado'] ?? 'confirmada') === 'confirmada') {
                $stockModel = new Stock();
                foreach ($items as $item) {
                    $stockModel->ajustar(
                        $item['producto_id'],
                        $venta['sucursal_id'],
                        $item['cantidad'],
                        'venta',
                        "Venta $correlativo"
                    );
                }

                // Registrar en caja si hay sesión abierta
                $sesion = $db->fetchOne(
                    "SELECT id FROM caja_sesiones WHERE sucursal_id = ? AND estado = 'abierta' LIMIT 1",
                    [$venta['sucursal_id']]
                );
                if ($sesion) {
                    foreach ($pagos as $pago) {
                        if ($pago['monto'] > 0) {
                            $db->insert('caja_movimientos', [
                                'empresa_id'     => $this->empresaScope(),
                                'sucursal_id'    => $venta['sucursal_id'],
                                'sesion_id'      => $sesion['id'],
                                'usuario_id'     => Auth::id(),
                                'tipo'           => 'ingreso',
                                'concepto'       => "Venta $correlativo",
                                'monto'          => $pago['monto'],
                                'forma_pago_id'  => $pago['forma_pago_id'],
                                'referencia_tipo'=> 'venta',
                                'referencia_id'  => $ventaId,
                                'fecha'          => date('Y-m-d H:i:s'),
                            ]);
                        }
                    }
                }
            }

            AuditLog::log('crear_venta', 'ventas', $ventaId, null, ['correlativo' => $correlativo, 'total' => $total]);
            $db->commit();
            return $ventaId;
        } catch (Exception $e) {
            $db->rollBack();
            error_log('Venta::crear error: ' . $e->getMessage());
            throw $e;
        }
    }

    public function anular(int $id, string $motivo = ''): bool
    {
        $venta = $this->getById($id);
        if (!$venta || $venta['estado'] !== 'confirmada') return false;

        $db = $this->db;
        try {
            $db->beginTransaction();
            $db->update('ventas', ['estado' => 'anulada', 'observaciones' => $motivo], ['id' => $id]);

            // Restaurar stock
            $items = $this->getItems($id);
            $stockModel = new Stock();
            foreach ($items as $item) {
                $stockModel->ajustar($item['producto_id'], $venta['sucursal_id'], $item['cantidad'], 'entrada', "Anulación venta {$venta['correlativo']}");
            }

            AuditLog::log('anular_venta', 'ventas', $id, $venta, ['estado' => 'anulada']);
            $db->commit();
            return true;
        } catch (Exception $e) {
            $db->rollBack();
            return false;
        }
    }

    public function getResumenDiario(?int $sucursalId = null, string $fecha = ''): array
    {
        $fecha    = $fecha ?: date('Y-m-d');
        $params   = [$this->empresaScope(), $fecha];
        $sucWhere = '';
        if ($sucursalId) { $sucWhere = 'AND v.sucursal_id = ?'; $params[] = $sucursalId; }

        return $this->db->fetchOne(
            "SELECT COUNT(*) as total_ventas, COALESCE(SUM(v.total),0) as total_monto,
                    COALESCE(SUM(v.descuento),0) as total_descuento
             FROM ventas v
             WHERE v.empresa_id = ? AND DATE(v.fecha) = ? AND v.estado = 'confirmada' $sucWhere",
            $params
        ) ?: ['total_ventas' => 0, 'total_monto' => 0, 'total_descuento' => 0];
    }

    public function getFormasPago(): array
    {
        return $this->db->fetchAll(
            "SELECT * FROM formas_pago WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$this->empresaScope()]
        );
    }

    public function getVentasPorMes(int $year): array
    {
        return $this->db->fetchAll(
            "SELECT MONTH(fecha) as mes, COUNT(*) as total_ventas, SUM(total) as total_monto
             FROM ventas
             WHERE empresa_id = ? AND YEAR(fecha) = ? AND estado = 'confirmada'
             GROUP BY MONTH(fecha) ORDER BY mes",
            [$this->empresaScope(), $year]
        );
    }
}
