<?php
class Sucursal extends BaseModel
{
    protected string $table = 'sucursales';

    public function getByEmpresa(int $empresaId): array
    {
        return $this->db->fetchAll(
            "SELECT * FROM sucursales WHERE empresa_id = ? AND estado = 'activa' ORDER BY es_principal DESC, nombre",
            [$empresaId]
        );
    }

    public function getAll(int $page = 1): array
    {
        // SuperAdmin ve todas las sucursales de todas las empresas
        if (Auth::isSuperAdmin()) {
            $sql = "SELECT s.*, e.nombre as empresa_nombre,
                           (SELECT COUNT(*) FROM usuarios WHERE sucursal_id = s.id) as total_usuarios
                    FROM sucursales s
                    LEFT JOIN empresas e ON s.empresa_id = e.id
                    ORDER BY e.nombre ASC, s.es_principal DESC, s.nombre";
            return $this->db->paginate($sql, [], $page);
        }

        $params = [$this->empresaScope()];
        $sql = "SELECT s.*, e.nombre as empresa_nombre,
                       (SELECT COUNT(*) FROM usuarios WHERE sucursal_id = s.id) as total_usuarios
                FROM sucursales s
                LEFT JOIN empresas e ON s.empresa_id = e.id
                WHERE s.empresa_id = ? ORDER BY s.es_principal DESC, s.nombre";
        return $this->db->paginate($sql, $params, $page);
    }

    public function getById(int $id): array|false
    {
        // SuperAdmin puede ver cualquier sucursal sin filtro de empresa
        if (Auth::isSuperAdmin()) {
            return $this->db->fetchOne(
                "SELECT s.*, e.nombre as empresa_nombre
                 FROM sucursales s LEFT JOIN empresas e ON s.empresa_id = e.id
                 WHERE s.id = ?",
                [$id]
            );
        }
        return $this->db->fetchOne(
            "SELECT s.*, e.nombre as empresa_nombre
             FROM sucursales s LEFT JOIN empresas e ON s.empresa_id = e.id
             WHERE s.id = ? AND s.empresa_id = ?",
            [$id, $this->empresaScope()]
        );
    }

    // ── Crear sucursal ────────────────────────────────────────
    public function crear(array $data): int
    {
        // SuperAdmin puede especificar empresa_id desde el formulario
        $empresaId = isset($data['empresa_id']) && (int)$data['empresa_id'] > 0
            ? (int)$data['empresa_id']
            : $this->empresaScope();

        return $this->db->insert('sucursales', [
            'empresa_id'  => $empresaId,
            'nombre'      => trim($data['nombre']),
            'codigo'      => trim($data['codigo'] ?? '') ?: null,
            'direccion'   => trim($data['direccion'] ?? '') ?: null,
            'telefono'    => trim($data['telefono'] ?? '') ?: null,
            'email'       => trim($data['email'] ?? '') ?: null,
            'horario'     => trim($data['horario'] ?? '') ?: null,
            'es_principal'=> 0,
            'estado'      => $data['estado'] ?? 'activa',
        ]);
    }
}
