<?php
/**
 * OptiCore SaaS - Licencia Model
 * Gestión de licencias y suscripciones por empresa
 */

class Licencia extends BaseModel
{
    protected string $table = 'licencias';

    // ── Listar licencias (SuperAdmin: todas | Admin: su empresa) ──
    public function getAll(): array
    {
        if (Auth::isSuperAdmin()) {
            return $this->db->fetchAll(
                "SELECT l.*, e.nombre as empresa_nombre, p.nombre as plan_nombre
                 FROM licencias l
                 JOIN empresas e ON l.empresa_id = e.id
                 JOIN planes p ON l.plan_id = p.id
                 ORDER BY l.created_at DESC"
            );
        }
        return $this->db->fetchAll(
            "SELECT l.*, e.nombre as empresa_nombre, p.nombre as plan_nombre
             FROM licencias l
             JOIN empresas e ON l.empresa_id = e.id
             JOIN planes p ON l.plan_id = p.id
             WHERE l.empresa_id = ?
             ORDER BY l.created_at DESC",
            [$this->empresaScope()]
        );
    }

    // ── Licencias de una empresa ──────────────────────────────
    public function getByEmpresa(int $empresaId): array
    {
        return $this->db->fetchAll(
            "SELECT l.*, p.nombre as plan_nombre
             FROM licencias l
             JOIN planes p ON l.plan_id = p.id
             WHERE l.empresa_id = ?
             ORDER BY l.created_at DESC",
            [$empresaId]
        );
    }

    // ── Licencia activa de una empresa ────────────────────────
    public function getLicenciaActiva(int $empresaId): array|false
    {
        return $this->db->fetchOne(
            "SELECT l.*, p.nombre as plan_nombre, p.max_sucursales,
                    p.max_usuarios, p.max_productos, p.max_pacientes,
                    p.tiene_laboratorio, p.tiene_agenda, p.tiene_reportes, p.tiene_api
             FROM licencias l
             JOIN planes p ON l.plan_id = p.id
             WHERE l.empresa_id = ? AND l.estado = 'activa'
             ORDER BY l.fecha_fin DESC
             LIMIT 1",
            [$empresaId]
        );
    }

    // ── Obtener por ID ────────────────────────────────────────
    public function getById(int $id): array|false
    {
        return $this->db->fetchOne(
            "SELECT l.*, e.nombre as empresa_nombre, p.nombre as plan_nombre
             FROM licencias l
             JOIN empresas e ON l.empresa_id = e.id
             JOIN planes p ON l.plan_id = p.id
             WHERE l.id = ?",
            [$id]
        );
    }

    // ── Crear licencia ────────────────────────────────────────
    public function crear(array $data): int
    {
        return $this->db->insert('licencias', [
            'empresa_id'    => $data['empresa_id'],
            'plan_id'       => $data['plan_id'],
            'tipo'          => $data['tipo'] ?? 'mensual',
            'fecha_inicio'  => $data['fecha_inicio'],
            'fecha_fin'     => $data['fecha_fin'] ?? null,
            'monto'         => $data['monto'] ?? 0,
            'estado'        => $data['estado'] ?? 'activa',
            'observaciones' => $data['observaciones'] ?? null,
        ]);
    }

    // ── Actualizar licencia ───────────────────────────────────
    public function actualizar(int $id, array $data): int
    {
        return $this->db->update('licencias', [
            'plan_id'       => $data['plan_id'],
            'tipo'          => $data['tipo'] ?? 'mensual',
            'fecha_inicio'  => $data['fecha_inicio'],
            'fecha_fin'     => $data['fecha_fin'] ?? null,
            'monto'         => $data['monto'] ?? 0,
            'estado'        => $data['estado'] ?? 'activa',
            'observaciones' => $data['observaciones'] ?? null,
        ], ['id' => $id]);
    }

    // ── Suspender licencia ────────────────────────────────────
    public function suspender(int $id): void
    {
        $this->db->update('licencias', ['estado' => 'suspendida'], ['id' => $id]);
    }

    // ── Cancelar licencia ─────────────────────────────────────
    public function cancelar(int $id): void
    {
        $this->db->update('licencias', ['estado' => 'cancelada'], ['id' => $id]);
    }

    // ── Licencias próximas a vencer (en X días) ───────────────
    public function getProximasVencer(int $dias = 7): array
    {
        return $this->db->fetchAll(
            "SELECT l.*, e.nombre as empresa_nombre, e.email as empresa_email,
                    p.nombre as plan_nombre,
                    DATEDIFF(l.fecha_fin, CURDATE()) as dias_restantes
             FROM licencias l
             JOIN empresas e ON l.empresa_id = e.id
             JOIN planes p ON l.plan_id = p.id
             WHERE l.estado = 'activa'
               AND l.fecha_fin IS NOT NULL
               AND l.fecha_fin BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL ? DAY)
             ORDER BY l.fecha_fin ASC",
            [$dias]
        );
    }

    // ── Expirar licencias vencidas ────────────────────────────
    public function expirarVencidas(): int
    {
        $stmt = $this->db->query(
            "UPDATE licencias SET estado = 'expirada'
             WHERE estado = 'activa'
               AND fecha_fin IS NOT NULL
               AND fecha_fin < CURDATE()"
        );
        return $stmt->rowCount();
    }

    // ── Estadísticas de licencias ─────────────────────────────
    public function getEstadisticas(): array
    {
        return $this->db->fetchOne(
            "SELECT
                COUNT(*) as total,
                SUM(estado = 'activa') as activas,
                SUM(estado = 'suspendida') as suspendidas,
                SUM(estado = 'expirada') as expiradas,
                SUM(estado = 'cancelada') as canceladas,
                SUM(monto) as ingresos_totales
             FROM licencias"
        ) ?: [];
    }
}
