<?php
/**
 * OptiCore SaaS - UsuarioController
 */

class UsuarioController
{
    private Usuario $model;

    public function __construct()
    {
        Auth::requirePermission('usuarios.ver');
        $this->model = new Usuario();
    }

    // ── GET /usuarios ─────────────────────────────────────────
    public function index(): void
    {
        $page     = currentPage();
        $busqueda = get('q');
        $result   = $this->model->getAll($page, $busqueda);
        view('usuarios.index', [
            'title'      => 'Usuarios',
            'usuarios'   => $result['data'],
            'pagination' => $result,
            'busqueda'   => $busqueda,
        ]);
    }

    // ── GET /usuarios/create ──────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('usuarios.crear');

        $empresaId = Auth::empresaId();
        if ($empresaId && !(new Plan())->verificarLimite($empresaId, 'usuarios')) {
            flash('error', 'Has alcanzado el límite de usuarios de tu plan.');
            redirect('/usuarios');
        }

        $roles      = $this->getRolesDisponibles();
        $sucursales = $this->getSucursalesDisponibles();

        view('usuarios.create', [
            'title'      => 'Nuevo Usuario',
            'roles'      => $roles,
            'sucursales' => $sucursales,
        ]);
    }

    // ── POST /usuarios ────────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('usuarios.crear');
        csrf_verify();

        $errors = validateRequired($_POST, [
            'nombre'   => 'Nombre',
            'apellido' => 'Apellido',
            'email'    => 'Email',
            'password' => 'Contraseña',
            'rol_id'   => 'Rol',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/usuarios/create');
        }

        if (!filter_var($_POST['email'], FILTER_VALIDATE_EMAIL)) {
            flash('error', 'El email no es válido.');
            redirect('/usuarios/create');
        }

        if (strlen($_POST['password']) < 6) {
            flash('error', 'La contraseña debe tener al menos 6 caracteres.');
            redirect('/usuarios/create');
        }

        // Verificar email único
        $existe = db()->fetchOne("SELECT id FROM usuarios WHERE email = ?", [$_POST['email']]);
        if ($existe) {
            flash('error', 'El email ya está registrado.');
            redirect('/usuarios/create');
        }

        $empresaId = Auth::isSuperAdmin()
            ? (int) ($_POST['empresa_id'] ?? 0) ?: null
            : Auth::empresaId();

        $id = db()->insert('usuarios', [
            'empresa_id'  => $empresaId,
            'sucursal_id' => !empty($_POST['sucursal_id']) ? (int) $_POST['sucursal_id'] : null,
            'rol_id'      => (int) $_POST['rol_id'],
            'nombre'      => trim($_POST['nombre']),
            'apellido'    => trim($_POST['apellido']),
            'email'       => strtolower(trim($_POST['email'])),
            'password'    => password_hash($_POST['password'], PASSWORD_BCRYPT, ['cost' => BCRYPT_COST]),
            'telefono'    => $_POST['telefono'] ?? null,
            'estado'      => $_POST['estado'] ?? 'activo',
        ]);

        AuditLog::log('crear_usuario', 'usuarios', $id);
        flash('success', 'Usuario creado correctamente.');
        redirect('/usuarios');
    }

    // ── GET /usuarios/{id}/edit ───────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('usuarios.editar');

        $usuario = $this->model->findById($id);
        if (!$usuario) {
            flash('error', 'Usuario no encontrado.');
            redirect('/usuarios');
        }

        $roles      = $this->getRolesDisponibles();
        $sucursales = $this->getSucursalesDisponibles();

        view('usuarios.edit', [
            'title'      => 'Editar Usuario',
            'usuario'    => $usuario,
            'roles'      => $roles,
            'sucursales' => $sucursales,
        ]);
    }

    // ── POST /usuarios/{id} ───────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('usuarios.editar');
        csrf_verify();

        $usuario = $this->model->findById($id);
        if (!$usuario) {
            flash('error', 'Usuario no encontrado.');
            redirect('/usuarios');
        }

        $errors = validateRequired($_POST, [
            'nombre'   => 'Nombre',
            'apellido' => 'Apellido',
            'email'    => 'Email',
            'rol_id'   => 'Rol',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/usuarios/$id/edit");
        }

        // Verificar email único (excluyendo el actual)
        $existe = db()->fetchOne(
            "SELECT id FROM usuarios WHERE email = ? AND id != ?",
            [$_POST['email'], $id]
        );
        if ($existe) {
            flash('error', 'El email ya está en uso por otro usuario.');
            redirect("/usuarios/$id/edit");
        }

        $updateData = [
            'sucursal_id' => !empty($_POST['sucursal_id']) ? (int) $_POST['sucursal_id'] : null,
            'rol_id'      => (int) $_POST['rol_id'],
            'nombre'      => trim($_POST['nombre']),
            'apellido'    => trim($_POST['apellido']),
            'email'       => strtolower(trim($_POST['email'])),
            'telefono'    => $_POST['telefono'] ?? null,
            'estado'      => $_POST['estado'] ?? 'activo',
        ];

        // Actualizar contraseña solo si se proporcionó
        if (!empty($_POST['password'])) {
            if (strlen($_POST['password']) < 6) {
                flash('error', 'La contraseña debe tener al menos 6 caracteres.');
                redirect("/usuarios/$id/edit");
            }
            $updateData['password'] = password_hash($_POST['password'], PASSWORD_BCRYPT, ['cost' => BCRYPT_COST]);
        }

        $this->model->update($id, $updateData);
        AuditLog::log('editar_usuario', 'usuarios', $id);
        flash('success', 'Usuario actualizado correctamente.');
        redirect('/usuarios');
    }

    // ── POST /usuarios/{id}/delete ────────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('usuarios.eliminar');
        csrf_verify();

        // No puede eliminarse a sí mismo
        if ($id === Auth::id()) {
            flash('error', 'No puedes eliminar tu propia cuenta.');
            redirect('/usuarios');
        }

        $usuario = $this->model->findById($id);
        if (!$usuario) {
            flash('error', 'Usuario no encontrado.');
            redirect('/usuarios');
        }

        $this->model->update($id, ['estado' => 'inactivo']);
        AuditLog::log('desactivar_usuario', 'usuarios', $id, $usuario);
        flash('success', 'Usuario desactivado correctamente.');
        redirect('/usuarios');
    }

    // ── Roles disponibles según contexto ─────────────────────
    private function getRolesDisponibles(): array
    {
        if (Auth::isSuperAdmin()) {
            return db()->fetchAll("SELECT * FROM roles WHERE estado = 'activo' ORDER BY nombre");
        }
        $empresaId = Auth::empresaId();
        return db()->fetchAll(
            "SELECT * FROM roles
             WHERE (empresa_id = ? OR empresa_id IS NULL)
               AND slug != 'superadmin' AND estado = 'activo'
             ORDER BY nombre",
            [$empresaId]
        );
    }

    // ── Sucursales disponibles ────────────────────────────────
    private function getSucursalesDisponibles(): array
    {
        if (Auth::isSuperAdmin()) return [];
        $empresaId = Auth::empresaId();
        return db()->fetchAll(
            "SELECT * FROM sucursales WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$empresaId]
        );
    }
}
