<?php
/**
 * OptiCore SaaS - StockController
 */

class StockController
{
    private Stock $model;

    public function __construct()
    {
        Auth::requirePermission('stock.ver');
        $this->model = new Stock();
    }

    // ── GET /stock ────────────────────────────────────────────
    public function index(): void
    {
        $sucursalId = getInt('sucursal_id') ?: Auth::sucursalId();
        $busqueda   = get('q');
        $page       = currentPage();

        $resultado  = $this->model->listarStock($sucursalId, $busqueda ?: null, $page, ITEMS_PER_PAGE);
        $sucursales = $this->getSucursales();

        view('stock.index', [
            'title'       => 'Stock',
            'stock'       => $resultado['data'],
            'pagination'  => $resultado['pagination'],
            'sucursales'  => $sucursales,
            'sucursal_id' => $sucursalId,
            'busqueda'    => $busqueda,
        ]);
    }

    // ── GET /stock/movimientos ────────────────────────────────
    public function movimientos(): void
    {
        $sucursalId  = getInt('sucursal_id') ?: Auth::sucursalId();
        $productoId  = getInt('producto_id');
        $page        = currentPage();

        $resultado   = $this->model->listarMovimientos(
            $sucursalId,
            $productoId ?: null,
            $page,
            ITEMS_PER_PAGE
        );
        $sucursales  = $this->getSucursales();

        view('stock.movimientos', [
            'title'       => 'Movimientos de Stock',
            'movimientos' => $resultado['data'],
            'pagination'  => $resultado['pagination'],
            'sucursales'  => $sucursales,
            'sucursal_id' => $sucursalId,
            'producto_id' => $productoId,
        ]);
    }

    // ── POST /stock/ajuste ────────────────────────────────────
    public function ajuste(): void
    {
        Auth::requirePermission('stock.ajustar');
        csrf_verify();

        $errors = validateRequired($_POST, [
            'producto_id'  => 'Producto',
            'sucursal_id'  => 'Sucursal',
            'cantidad'     => 'Cantidad',
            'tipo'         => 'Tipo de ajuste',
            'observaciones'=> 'Observaciones',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/stock');
        }

        $productoId  = (int) $_POST['producto_id'];
        $sucursalId  = (int) $_POST['sucursal_id'];
        $cantidad    = (int) $_POST['cantidad'];
        $tipo        = $_POST['tipo'];
        $obs         = $_POST['observaciones'];

        if ($cantidad <= 0) {
            flash('error', 'La cantidad debe ser mayor a 0.');
            redirect('/stock');
        }

        try {
            $this->model->ajustar($productoId, $sucursalId, $cantidad, $tipo, $obs, Auth::id());
            AuditLog::log('ajuste_stock', 'stock', $productoId, null, $_POST);
            flash('success', 'Ajuste de stock registrado correctamente.');
        } catch (Exception $e) {
            flash('error', 'Error al ajustar stock: ' . $e->getMessage());
        }

        redirect('/stock');
    }

    // ── GET /stock/kardex?producto_id={id} ────────────────────
    public function kardex(): void
    {
        Auth::requirePermission('stock.kardex');

        $productoId = getInt('producto_id');
        if (!$productoId) {
            flash('error', 'Debes especificar un producto para ver el kardex.');
            redirect('/stock');
        }

        $producto = (new Producto())->findById($productoId);
        if (!$producto) {
            flash('error', 'Producto no encontrado.');
            redirect('/stock');
        }

        $sucursalId = getInt('sucursal_id') ?: Auth::sucursalId();
        $kardex     = $this->model->getKardex($productoId, $sucursalId);
        $sucursales = $this->getSucursales();

        view('stock.kardex', [
            'title'       => 'Kardex: ' . $producto['nombre'],
            'producto'    => $producto,
            'kardex'      => $kardex,
            'sucursales'  => $sucursales,
            'sucursal_id' => $sucursalId,
        ]);
    }

    // ── Sucursales disponibles ────────────────────────────────
    private function getSucursales(): array
    {
        $empresaId = Auth::empresaId();
        if (!$empresaId) return [];
        return db()->fetchAll(
            "SELECT id, nombre FROM sucursales WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$empresaId]
        );
    }
}
