<?php
/**
 * OptiCore SaaS - RolController
 */

class RolController
{
    private Rol $model;

    public function __construct()
    {
        Auth::requirePermission('roles.ver');
        $this->model = new Rol();
    }

    // ── GET /roles ────────────────────────────────────────────
    public function index(): void
    {
        $roles = $this->model->getAll();
        view('roles.index', ['title' => 'Roles y Permisos', 'roles' => $roles]);
    }

    // ── GET /roles/create ─────────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('roles.crear');
        $permisos = $this->model->getTodosPermisos();
        view('roles.create', ['title' => 'Nuevo Rol', 'permisos' => $permisos]);
    }

    // ── POST /roles ───────────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('roles.crear');
        csrf_verify();

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/roles/create');
        }

        $empresaId = Auth::isSuperAdmin() ? null : Auth::empresaId();
        $slug      = strtolower(preg_replace('/[^a-zA-Z0-9]+/', '_', trim($_POST['nombre'])));

        $id = db()->insert('roles', [
            'empresa_id'  => $empresaId,
            'nombre'      => trim($_POST['nombre']),
            'slug'        => $slug,
            'descripcion' => $_POST['descripcion'] ?? null,
            'es_sistema'  => 0,
            'estado'      => 'activo',
        ]);

        // Sincronizar permisos
        $permisoIds = $_POST['permisos'] ?? [];
        if (!empty($permisoIds)) {
            $this->model->sincronizarPermisos($id, array_map('intval', $permisoIds));
        }

        AuditLog::log('crear_rol', 'roles', $id, null, $_POST);
        flash('success', 'Rol creado correctamente.');
        redirect('/roles');
    }

    // ── GET /roles/{id}/edit ──────────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('roles.editar');

        $rol = $this->model->getById($id);
        if (!$rol) {
            flash('error', 'Rol no encontrado.');
            redirect('/roles');
        }

        if ($rol['es_sistema'] && !Auth::isSuperAdmin()) {
            flash('error', 'No puedes editar roles del sistema.');
            redirect('/roles');
        }

        $permisos = $this->model->getPermisosAgrupados($id);
        view('roles.edit', [
            'title'    => 'Editar Rol',
            'rol'      => $rol,
            'permisos' => $permisos,
        ]);
    }

    // ── POST /roles/{id} ──────────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('roles.editar');
        csrf_verify();

        $rol = $this->model->getById($id);
        if (!$rol) {
            flash('error', 'Rol no encontrado.');
            redirect('/roles');
        }

        if ($rol['es_sistema'] && !Auth::isSuperAdmin()) {
            flash('error', 'No puedes editar roles del sistema.');
            redirect('/roles');
        }

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/roles/$id/edit");
        }

        db()->update('roles', [
            'nombre'      => trim($_POST['nombre']),
            'descripcion' => $_POST['descripcion'] ?? null,
            'estado'      => $_POST['estado'] ?? 'activo',
        ], ['id' => $id]);

        // Sincronizar permisos
        $permisoIds = $_POST['permisos'] ?? [];
        $this->model->sincronizarPermisos($id, array_map('intval', $permisoIds));

        // Refrescar permisos en sesión si el usuario tiene este rol
        if (Auth::user()['rol_id'] === $id) {
            Auth::refreshPermissions();
        }

        AuditLog::log('editar_rol', 'roles', $id, $rol, $_POST);
        flash('success', 'Rol actualizado correctamente.');
        redirect('/roles');
    }

    // ── POST /roles/{id}/delete ───────────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('roles.eliminar');
        csrf_verify();

        $rol = $this->model->getById($id);
        if (!$rol) {
            flash('error', 'Rol no encontrado.');
            redirect('/roles');
        }

        if ($rol['es_sistema']) {
            flash('error', 'No se pueden eliminar roles del sistema.');
            redirect('/roles');
        }

        // Verificar que no tenga usuarios asignados
        $count = db()->count('usuarios', ['rol_id' => $id]);
        if ($count > 0) {
            flash('error', "No se puede eliminar el rol porque tiene $count usuario(s) asignado(s).");
            redirect('/roles');
        }

        db()->delete('roles', ['id' => $id]);
        AuditLog::log('eliminar_rol', 'roles', $id, $rol);
        flash('success', 'Rol eliminado correctamente.');
        redirect('/roles');
    }
}
