<?php
/**
 * OptiCore SaaS - ReporteController
 * Reportes de Ventas, Stock y Caja con exportación CSV
 */

class ReporteController
{
    public function __construct()
    {
        Auth::requirePermission('reportes.ver');
    }

    // ── GET /reportes ─────────────────────────────────────────
    public function index(): void
    {
        $empresaId  = Auth::empresaId();
        $sucursalId = Auth::sucursalId();
        $db         = db();

        $sucWhere = $sucursalId ? " AND sucursal_id = $sucursalId" : "";

        $resumen = [
            'ventas_mes' => $db->fetchOne(
                "SELECT COALESCE(SUM(total),0) as v FROM ventas
                 WHERE empresa_id = ? AND estado = 'confirmada'
                   AND DATE_FORMAT(fecha,'%Y-%m') = DATE_FORMAT(NOW(),'%Y-%m')$sucWhere",
                [$empresaId]
            )['v'] ?? 0,

            'cantidad_ventas_mes' => $db->fetchOne(
                "SELECT COUNT(*) as v FROM ventas
                 WHERE empresa_id = ? AND estado = 'confirmada'
                   AND DATE_FORMAT(fecha,'%Y-%m') = DATE_FORMAT(NOW(),'%Y-%m')$sucWhere",
                [$empresaId]
            )['v'] ?? 0,

            'productos_bajo_stock' => $db->fetchOne(
                "SELECT COUNT(*) as v FROM stock s
                 JOIN productos p ON s.producto_id = p.id
                 WHERE s.empresa_id = ? AND s.cantidad <= p.stock_minimo AND p.estado = 'activo'",
                [$empresaId]
            )['v'] ?? 0,

            'pacientes_mes' => $db->fetchOne(
                "SELECT COUNT(*) as v FROM pacientes
                 WHERE empresa_id = ?
                   AND DATE_FORMAT(created_at,'%Y-%m') = DATE_FORMAT(NOW(),'%Y-%m')",
                [$empresaId]
            )['v'] ?? 0,
        ];

        view('reportes.index', ['title' => 'Reportes', 'resumen' => $resumen]);
    }

    // ── GET /reportes/ventas ──────────────────────────────────
    public function ventas(): void
    {
        Auth::requirePermission('reportes.ventas');

        $empresaId  = Auth::empresaId();
        $sucursalId = getInt('sucursal_id') ?: Auth::sucursalId();
        $desde      = get('desde') ?: date('Y-m-01');
        $hasta      = get('hasta') ?: date('Y-m-d');
        $agrupar    = get('agrupar') ?: 'dia';

        $db       = db();
        $sucWhere = $sucursalId ? " AND sucursal_id = $sucursalId" : "";

        // KPIs
        $raw = $db->fetchOne(
            "SELECT COUNT(*) as cantidad,
                    COALESCE(SUM(total),0) as total,
                    COALESCE(SUM(descuento),0) as descuentos,
                    COALESCE(AVG(total),0) as promedio
             FROM ventas
             WHERE empresa_id = ? AND estado = 'confirmada'
               AND DATE(fecha) BETWEEN ? AND ?$sucWhere",
            [$empresaId, $desde, $hasta]
        );

        $kpis = [
            'total'      => $raw['total']     ?? 0,
            'cantidad'   => $raw['cantidad']  ?? 0,
            'promedio'   => $raw['promedio']  ?? 0,
            'descuentos' => $raw['descuentos'] ?? 0,
        ];

        // Agrupación
        $groupExpr = match($agrupar) {
            'mes'    => "DATE_FORMAT(fecha,'%Y-%m')",
            'semana' => "YEARWEEK(fecha,1)",
            default  => "DATE(fecha)",
        };

        $datos = $db->fetchAll(
            "SELECT $groupExpr as periodo,
                    COUNT(*) as cantidad,
                    COALESCE(SUM(subtotal),0) as subtotal,
                    COALESCE(SUM(descuento),0) as descuentos,
                    COALESCE(SUM(total),0) as total,
                    COALESCE(AVG(total),0) as promedio
             FROM ventas
             WHERE empresa_id = ? AND estado = 'confirmada'
               AND DATE(fecha) BETWEEN ? AND ?$sucWhere
             GROUP BY periodo ORDER BY periodo ASC",
            [$empresaId, $desde, $hasta]
        );

        $sucursales = $this->getSucursales($empresaId);

        view('reportes.ventas', [
            'title'       => 'Reporte de Ventas',
            'kpis'        => $kpis,
            'datos'       => $datos,
            'sucursales'  => $sucursales,
            'sucursal_id' => $sucursalId,
            'desde'       => $desde,
            'hasta'       => $hasta,
            'agrupar'     => $agrupar,
        ]);
    }

    // ── GET /reportes/ventas/export ───────────────────────────
    public function exportVentas(): void
    {
        Auth::requirePermission('reportes.ventas');

        $empresaId  = Auth::empresaId();
        $sucursalId = getInt('sucursal_id') ?: Auth::sucursalId();
        $desde      = get('desde') ?: date('Y-m-01');
        $hasta      = get('hasta') ?: date('Y-m-d');
        $db         = db();
        $sucWhere   = $sucursalId ? " AND v.sucursal_id = $sucursalId" : "";

        $rows = $db->fetchAll(
            "SELECT v.correlativo,
                    DATE(v.fecha) as fecha,
                    CONCAT(COALESCE(p.nombre,''), ' ', COALESCE(p.apellido,'')) as paciente,
                    CONCAT(u.nombre,' ',u.apellido) as vendedor,
                    su.nombre as sucursal,
                    v.subtotal, v.descuento, v.total, v.estado
             FROM ventas v
             LEFT JOIN pacientes p ON v.paciente_id = p.id
             LEFT JOIN usuarios u ON v.usuario_id = u.id
             LEFT JOIN sucursales su ON v.sucursal_id = su.id
             WHERE v.empresa_id = ? AND v.estado = 'confirmada'
               AND DATE(v.fecha) BETWEEN ? AND ?$sucWhere
             ORDER BY v.fecha DESC",
            [$empresaId, $desde, $hasta]
        );

        $this->streamCsv(
            'reporte_ventas_' . $desde . '_' . $hasta . '.csv',
            ['Correlativo', 'Fecha', 'Paciente', 'Vendedor', 'Sucursal', 'Subtotal', 'Descuento', 'Total', 'Estado'],
            array_map(fn($r) => [
                $r['correlativo'],
                $r['fecha'],
                trim($r['paciente']) ?: 'Sin paciente',
                $r['vendedor'],
                $r['sucursal'],
                $r['subtotal'],
                $r['descuento'],
                $r['total'],
                $r['estado'],
            ], $rows)
        );
    }

    // ── GET /reportes/stock ───────────────────────────────────
    public function stock(): void
    {
        Auth::requirePermission('reportes.stock');

        $empresaId  = Auth::empresaId();
        $sucursalId = getInt('sucursal_id') ?: Auth::sucursalId();
        $alerta     = get('alerta');
        $db         = db();

        $sucWhere   = $sucursalId ? " AND s.sucursal_id = $sucursalId" : "";
        $alertWhere = match($alerta) {
            'bajo'  => " AND s.cantidad > 0 AND s.cantidad <= p.stock_minimo",
            'sin'   => " AND s.cantidad <= 0",
            'ok'    => " AND s.cantidad > p.stock_minimo",
            default => "",
        };

        $stock = $db->fetchAll(
            "SELECT p.nombre as producto_nombre, p.sku,
                    c.nombre as categoria,
                    su.nombre as sucursal_nombre,
                    s.cantidad, p.stock_minimo, p.precio_costo
             FROM stock s
             JOIN productos p ON s.producto_id = p.id
             LEFT JOIN categorias c ON p.categoria_id = c.id
             JOIN sucursales su ON s.sucursal_id = su.id
             WHERE s.empresa_id = ? AND p.estado = 'activo'
               $sucWhere $alertWhere
             ORDER BY s.cantidad ASC, p.nombre ASC",
            [$empresaId]
        );

        $kpis = [
            'total_productos'  => count($stock),
            'bajo_stock'       => count(array_filter($stock, fn($s) => $s['cantidad'] > 0 && $s['cantidad'] <= $s['stock_minimo'])),
            'sin_stock'        => count(array_filter($stock, fn($s) => $s['cantidad'] <= 0)),
            'valor_inventario' => array_sum(array_map(fn($s) => ($s['precio_costo'] ?? 0) * $s['cantidad'], $stock)),
        ];

        // Categorías para filtro
        $catRows    = $db->fetchAll(
            "SELECT DISTINCT c.nombre FROM categorias c
             JOIN productos p ON p.categoria_id = c.id
             JOIN stock s ON s.producto_id = p.id
             WHERE s.empresa_id = ? AND p.estado = 'activo'
             ORDER BY c.nombre",
            [$empresaId]
        );
        $categorias = array_column($catRows, 'nombre');
        $sucursales = $this->getSucursales($empresaId);

        view('reportes.stock', [
            'title'       => 'Reporte de Stock',
            'stock'       => $stock,
            'kpis'        => $kpis,
            'categorias'  => $categorias,
            'sucursales'  => $sucursales,
            'sucursal_id' => $sucursalId,
        ]);
    }

    // ── GET /reportes/stock/export ────────────────────────────
    public function exportStock(): void
    {
        Auth::requirePermission('reportes.stock');

        $empresaId  = Auth::empresaId();
        $sucursalId = getInt('sucursal_id') ?: Auth::sucursalId();
        $db         = db();
        $sucWhere   = $sucursalId ? " AND s.sucursal_id = $sucursalId" : "";

        $rows = $db->fetchAll(
            "SELECT p.nombre as producto, p.sku,
                    c.nombre as categoria,
                    su.nombre as sucursal,
                    s.cantidad, p.stock_minimo, p.precio_costo,
                    (s.cantidad * p.precio_costo) as valor_total
             FROM stock s
             JOIN productos p ON s.producto_id = p.id
             LEFT JOIN categorias c ON p.categoria_id = c.id
             JOIN sucursales su ON s.sucursal_id = su.id
             WHERE s.empresa_id = ? AND p.estado = 'activo' $sucWhere
             ORDER BY p.nombre ASC",
            [$empresaId]
        );

        $this->streamCsv(
            'reporte_stock_' . date('Y-m-d') . '.csv',
            ['Producto', 'SKU', 'Categoría', 'Sucursal', 'Stock Actual', 'Stock Mínimo', 'Costo Unit.', 'Valor Total'],
            array_map(fn($r) => [
                $r['producto'],
                $r['sku'] ?? '',
                $r['categoria'] ?? 'Sin categoría',
                $r['sucursal'],
                $r['cantidad'],
                $r['stock_minimo'],
                $r['precio_costo'],
                $r['valor_total'],
            ], $rows)
        );
    }

    // ── GET /reportes/caja ────────────────────────────────────
    public function caja(): void
    {
        Auth::requirePermission('reportes.caja');

        $empresaId  = Auth::empresaId();
        $sucursalId = getInt('sucursal_id') ?: Auth::sucursalId();
        $desde      = get('desde') ?: date('Y-m-01');
        $hasta      = get('hasta') ?: date('Y-m-d');
        $db         = db();

        $sucWhere  = $sucursalId ? " AND cs.sucursal_id = $sucursalId" : "";
        $sucWhere2 = $sucursalId ? " AND v.sucursal_id = $sucursalId" : "";
        $sucWhere3 = $sucursalId ? " AND cs2.sucursal_id = $sucursalId" : "";

        // Cajas con total de ventas por sesión
        $cajas = $db->fetchAll(
            "SELECT cs.id, cs.fecha_apertura, cs.fecha_cierre,
                    su.nombre as sucursal_nombre,
                    cs.monto_apertura as saldo_inicial,
                    cs.monto_cierre_real as saldo_final,
                    cs.diferencia, cs.estado,
                    COALESCE((
                        SELECT SUM(cm.monto) FROM caja_movimientos cm
                        WHERE cm.sesion_id = cs.id AND cm.tipo = 'ingreso'
                    ), 0) as total_ventas
             FROM caja_sesiones cs
             JOIN sucursales su ON cs.sucursal_id = su.id
             WHERE cs.empresa_id = ?
               AND DATE(cs.fecha_apertura) BETWEEN ? AND ?$sucWhere
             ORDER BY cs.fecha_apertura DESC",
            [$empresaId, $desde, $hasta]
        );

        // KPIs
        $totalVentas = $db->fetchOne(
            "SELECT COALESCE(SUM(total),0) as v FROM ventas
             WHERE empresa_id = ? AND estado = 'confirmada'
               AND DATE(fecha) BETWEEN ? AND ?$sucWhere2",
            [$empresaId, $desde, $hasta]
        )['v'] ?? 0;

        $totalIngresos = $db->fetchOne(
            "SELECT COALESCE(SUM(cm.monto),0) as v FROM caja_movimientos cm
             JOIN caja_sesiones cs2 ON cm.sesion_id = cs2.id
             WHERE cs2.empresa_id = ? AND cm.tipo = 'ingreso'
               AND cm.referencia_tipo IS NULL
               AND DATE(cm.fecha) BETWEEN ? AND ?$sucWhere3",
            [$empresaId, $desde, $hasta]
        )['v'] ?? 0;

        $totalEgresos = $db->fetchOne(
            "SELECT COALESCE(SUM(cm.monto),0) as v FROM caja_movimientos cm
             JOIN caja_sesiones cs2 ON cm.sesion_id = cs2.id
             WHERE cs2.empresa_id = ? AND cm.tipo = 'egreso'
               AND DATE(cm.fecha) BETWEEN ? AND ?$sucWhere3",
            [$empresaId, $desde, $hasta]
        )['v'] ?? 0;

        $kpis = [
            'total_ventas'   => $totalVentas,
            'total_ingresos' => $totalIngresos,
            'total_egresos'  => $totalEgresos,
        ];

        // Por método de pago
        $porMetodo = $db->fetchAll(
            "SELECT fp.tipo as metodo, COUNT(*) as cantidad, SUM(vp.monto) as total
             FROM venta_pagos vp
             JOIN formas_pago fp ON vp.forma_pago_id = fp.id
             JOIN ventas v ON vp.venta_id = v.id
             WHERE v.empresa_id = ? AND v.estado = 'confirmada'
               AND DATE(v.fecha) BETWEEN ? AND ?$sucWhere2
             GROUP BY fp.tipo ORDER BY total DESC",
            [$empresaId, $desde, $hasta]
        );

        // Diferencias de caja
        $difRaw = $db->fetchOne(
            "SELECT COALESCE(SUM(CASE WHEN diferencia > 0 THEN diferencia ELSE 0 END),0) as sobrante,
                    COALESCE(SUM(CASE WHEN diferencia < 0 THEN ABS(diferencia) ELSE 0 END),0) as faltante
             FROM caja_sesiones
             WHERE empresa_id = ? AND estado = 'cerrada'
               AND DATE(fecha_apertura) BETWEEN ? AND ?" . ($sucursalId ? " AND sucursal_id = $sucursalId" : ""),
            [$empresaId, $desde, $hasta]
        );
        $diferencias = $difRaw ?: ['sobrante' => 0, 'faltante' => 0];

        $sucursales = $this->getSucursales($empresaId);

        view('reportes.caja', [
            'title'       => 'Reporte de Caja',
            'cajas'       => $cajas,
            'kpis'        => $kpis,
            'porMetodo'   => $porMetodo,
            'diferencias' => $diferencias,
            'sucursales'  => $sucursales,
            'sucursal_id' => $sucursalId,
            'desde'       => $desde,
            'hasta'       => $hasta,
        ]);
    }

    // ── GET /reportes/caja/export ─────────────────────────────
    public function exportCaja(): void
    {
        Auth::requirePermission('reportes.caja');

        $empresaId  = Auth::empresaId();
        $sucursalId = getInt('sucursal_id') ?: Auth::sucursalId();
        $desde      = get('desde') ?: date('Y-m-01');
        $hasta      = get('hasta') ?: date('Y-m-d');
        $db         = db();
        $sucWhere   = $sucursalId ? " AND cs.sucursal_id = $sucursalId" : "";

        $rows = $db->fetchAll(
            "SELECT cs.fecha_apertura, cs.fecha_cierre,
                    su.nombre as sucursal,
                    CONCAT(u.nombre,' ',u.apellido) as cajero,
                    cs.monto_apertura, cs.monto_cierre_sistema,
                    cs.monto_cierre_real, cs.diferencia, cs.estado
             FROM caja_sesiones cs
             JOIN sucursales su ON cs.sucursal_id = su.id
             JOIN usuarios u ON cs.usuario_id = u.id
             WHERE cs.empresa_id = ?
               AND DATE(cs.fecha_apertura) BETWEEN ? AND ?$sucWhere
             ORDER BY cs.fecha_apertura DESC",
            [$empresaId, $desde, $hasta]
        );

        $this->streamCsv(
            'reporte_caja_' . $desde . '_' . $hasta . '.csv',
            ['Apertura', 'Cierre', 'Sucursal', 'Cajero', 'Monto Apertura', 'Sistema', 'Real', 'Diferencia', 'Estado'],
            array_map(fn($r) => [
                $r['fecha_apertura'],
                $r['fecha_cierre'] ?? '',
                $r['sucursal'],
                $r['cajero'],
                $r['monto_apertura'],
                $r['monto_cierre_sistema'] ?? '',
                $r['monto_cierre_real'] ?? '',
                $r['diferencia'] ?? '',
                $r['estado'],
            ], $rows)
        );
    }

    // ── Helper: stream CSV al navegador ──────────────────────
    private function streamCsv(string $filename, array $headers, array $rows): void
    {
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        header('Expires: 0');

        $out = fopen('php://output', 'w');
        // UTF-8 BOM para que Excel lo abra correctamente
        fprintf($out, chr(0xEF) . chr(0xBB) . chr(0xBF));
        fputcsv($out, $headers, ';');
        foreach ($rows as $row) {
            fputcsv($out, $row, ';');
        }
        fclose($out);
        exit;
    }

    // ── Sucursales disponibles ────────────────────────────────
    private function getSucursales(int $empresaId): array
    {
        return db()->fetchAll(
            "SELECT id, nombre FROM sucursales WHERE empresa_id = ? AND estado = 'activa' ORDER BY nombre",
            [$empresaId]
        );
    }
}
