<?php
/**
 * OptiCore SaaS - PacienteController
 */

class PacienteController
{
    private Paciente $model;

    public function __construct()
    {
        Auth::requirePermission('pacientes.ver');
        $this->model = new Paciente();
    }

    // ── GET /pacientes ────────────────────────────────────────
    public function index(): void
    {
        $page      = currentPage();
        $busqueda  = get('q');
        $resultado = $this->model->buscar($busqueda, $page, ITEMS_PER_PAGE);

        view('pacientes.index', [
            'title'      => 'Pacientes',
            'pacientes'  => $resultado['data'],
            'pagination' => $resultado['pagination'],
            'busqueda'   => $busqueda,
        ]);
    }

    // ── GET /pacientes/buscar (AJAX) ──────────────────────────
    public function buscar(): void
    {
        $q         = get('q');
        $pacientes = $this->model->buscarSimple($q);
        jsonResponse($pacientes);
    }

    // ── GET /pacientes/create ─────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('pacientes.crear');

        $empresaId = Auth::empresaId();
        if ($empresaId && !(new Plan())->verificarLimite($empresaId, 'pacientes')) {
            flash('error', 'Has alcanzado el límite de pacientes de tu plan.');
            redirect('/pacientes');
        }

        view('pacientes.create', ['title' => 'Nuevo Paciente']);
    }

    // ── POST /pacientes ───────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('pacientes.crear');
        csrf_verify();

        $errors = validateRequired($_POST, [
            'nombre'   => 'Nombre',
            'apellido' => 'Apellido',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/pacientes/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_paciente', 'pacientes', $id);
        flash('success', 'Paciente registrado correctamente.');
        redirect("/pacientes/$id");
    }

    // ── GET /pacientes/{id} ───────────────────────────────────
    public function show(int $id): void
    {
        $paciente = $this->model->findById($id);
        if (!$paciente) {
            flash('error', 'Paciente no encontrado.');
            redirect('/pacientes');
        }

        $recetas = $this->model->getRecetas($id);
        $ventas  = $this->model->getVentas($id);

        view('pacientes.show', [
            'title'    => $paciente['nombre'] . ' ' . $paciente['apellido'],
            'paciente' => $paciente,
            'recetas'  => $recetas,
            'ventas'   => $ventas,
        ]);
    }

    // ── GET /pacientes/{id}/edit ──────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('pacientes.editar');

        $paciente = $this->model->findById($id);
        if (!$paciente) {
            flash('error', 'Paciente no encontrado.');
            redirect('/pacientes');
        }

        view('pacientes.edit', ['title' => 'Editar Paciente', 'paciente' => $paciente]);
    }

    // ── POST /pacientes/{id} ──────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('pacientes.editar');
        csrf_verify();

        $paciente = $this->model->findById($id);
        if (!$paciente) {
            flash('error', 'Paciente no encontrado.');
            redirect('/pacientes');
        }

        $errors = validateRequired($_POST, [
            'nombre'   => 'Nombre',
            'apellido' => 'Apellido',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/pacientes/$id/edit");
        }

        $this->model->actualizar($id, $_POST);
        AuditLog::log('editar_paciente', 'pacientes', $id, $paciente, $_POST);
        flash('success', 'Paciente actualizado correctamente.');
        redirect("/pacientes/$id");
    }

    // ── POST /pacientes/{id}/delete ───────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('pacientes.eliminar');
        csrf_verify();

        $paciente = $this->model->findById($id);
        if (!$paciente) {
            flash('error', 'Paciente no encontrado.');
            redirect('/pacientes');
        }

        $this->model->update($id, ['estado' => 'inactivo']);
        AuditLog::log('desactivar_paciente', 'pacientes', $id, $paciente);
        flash('success', 'Paciente desactivado correctamente.');
        redirect('/pacientes');
    }
}
