<?php
/**
 * OptiCore SaaS - MarcaController
 */

class MarcaController
{
    private Marca $model;

    public function __construct()
    {
        Auth::requirePermission('marcas.ver');
        $this->model = new Marca();
    }

    // ── GET /marcas ───────────────────────────────────────────
    public function index(): void
    {
        $page      = currentPage();
        $busqueda  = get('q');
        $resultado = $this->model->getAll($page, $busqueda);

        view('marcas.index', [
            'title'      => 'Marcas',
            'marcas'     => $resultado['data'],
            'pagination' => $resultado,
            'busqueda'   => $busqueda,
        ]);
    }

    // ── GET /marcas/create ────────────────────────────────────
    public function create(): void
    {
        Auth::requirePermission('marcas.crear');
        view('marcas.create', ['title' => 'Nueva Marca']);
    }

    // ── POST /marcas ──────────────────────────────────────────
    public function store(): void
    {
        Auth::requirePermission('marcas.crear');
        csrf_verify();

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/marcas/create');
        }

        $id = $this->model->crear($_POST);
        AuditLog::log('crear_marca', 'marcas', $id);
        flash('success', 'Marca creada correctamente.');
        redirect('/marcas');
    }

    // ── GET /marcas/{id}/edit ─────────────────────────────────
    public function edit(int $id): void
    {
        Auth::requirePermission('marcas.editar');

        $marca = $this->model->getById($id);
        if (!$marca) {
            flash('error', 'Marca no encontrada.');
            redirect('/marcas');
        }

        view('marcas.edit', [
            'title' => 'Editar Marca',
            'marca' => $marca,
        ]);
    }

    // ── POST /marcas/{id} ─────────────────────────────────────
    public function update(int $id): void
    {
        Auth::requirePermission('marcas.editar');
        csrf_verify();

        $marca = $this->model->getById($id);
        if (!$marca) {
            flash('error', 'Marca no encontrada.');
            redirect('/marcas');
        }

        $errors = validateRequired($_POST, ['nombre' => 'Nombre']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/marcas/$id/edit");
        }

        $this->model->actualizar($id, $_POST);
        AuditLog::log('editar_marca', 'marcas', $id, $marca, $_POST);
        flash('success', 'Marca actualizada correctamente.');
        redirect('/marcas');
    }

    // ── POST /marcas/{id}/delete ──────────────────────────────
    public function destroy(int $id): void
    {
        Auth::requirePermission('marcas.eliminar');
        csrf_verify();

        $marca = $this->model->getById($id);
        if (!$marca) {
            flash('error', 'Marca no encontrada.');
            redirect('/marcas');
        }

        if ($this->model->tieneProductos($id)) {
            flash('error', 'No se puede eliminar: la marca tiene productos asociados.');
            redirect('/marcas');
        }

        $this->model->eliminar($id);
        AuditLog::log('eliminar_marca', 'marcas', $id, $marca);
        flash('success', 'Marca desactivada correctamente.');
        redirect('/marcas');
    }
}
