<?php
/**
 * OptiCore SaaS - LicenciaController (Solo SuperAdmin)
 */

class LicenciaController
{
    private Licencia $model;

    public function __construct()
    {
        Auth::requireSuperAdmin();
        $this->model = new Licencia();
    }

    // ── GET /licencias ────────────────────────────────────────
    public function index(): void
    {
        $licencias   = $this->model->getAll();
        $estadisticas = $this->model->getEstadisticas();
        $porVencer   = $this->model->getProximasVencer(7);

        view('licencias.index', [
            'title'        => 'Licencias',
            'licencias'    => $licencias,
            'estadisticas' => $estadisticas,
            'por_vencer'   => $porVencer,
        ]);
    }

    // ── GET /licencias/create ─────────────────────────────────
    public function create(): void
    {
        $empresas = db()->fetchAll(
            "SELECT id, nombre FROM empresas WHERE estado != 'expirada' ORDER BY nombre"
        );
        $planes = (new Plan())->getActivos();

        view('licencias.create', [
            'title'    => 'Nueva Licencia',
            'empresas' => $empresas,
            'planes'   => $planes,
        ]);
    }

    // ── POST /licencias ───────────────────────────────────────
    public function store(): void
    {
        csrf_verify();

        $errors = validateRequired($_POST, [
            'empresa_id'   => 'Empresa',
            'plan_id'      => 'Plan',
            'tipo'         => 'Tipo',
            'fecha_inicio' => 'Fecha inicio',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/licencias/create');
        }

        // Calcular fecha fin automática según tipo
        $fechaFin = $_POST['fecha_fin'] ?? null;
        if (empty($fechaFin)) {
            $fechaFin = match($_POST['tipo']) {
                'mensual'  => date('Y-m-d', strtotime($_POST['fecha_inicio'] . ' +1 month')),
                'anual'    => date('Y-m-d', strtotime($_POST['fecha_inicio'] . ' +1 year')),
                'prueba'   => date('Y-m-d', strtotime($_POST['fecha_inicio'] . ' +30 days')),
                default    => null,
            };
        }

        $id = $this->model->crear(array_merge($_POST, ['fecha_fin' => $fechaFin]));

        // Actualizar estado de empresa a activa
        db()->update('empresas', ['estado' => 'activa'], ['id' => (int) $_POST['empresa_id']]);

        // Limpiar caché de licencia
        AuthMiddleware::clearLicenciaCache((int) $_POST['empresa_id']);

        AuditLog::log('crear_licencia', 'licencias', $id, null, $_POST);
        flash('success', 'Licencia creada correctamente.');
        redirect('/licencias');
    }

    // ── POST /licencias/{id}/suspender ────────────────────────
    public function suspender(int $id): void
    {
        csrf_verify();

        $licencia = $this->model->getById($id);
        if (!$licencia) {
            flash('error', 'Licencia no encontrada.');
            redirect('/licencias');
        }

        $this->model->suspender($id);
        AuthMiddleware::clearLicenciaCache((int) $licencia['empresa_id']);
        AuditLog::log('suspender_licencia', 'licencias', $id, $licencia);
        flash('success', 'Licencia suspendida.');
        redirect('/licencias');
    }

    // ── POST /licencias/{id}/cancelar ─────────────────────────
    public function cancelar(int $id): void
    {
        csrf_verify();

        $licencia = $this->model->getById($id);
        if (!$licencia) {
            flash('error', 'Licencia no encontrada.');
            redirect('/licencias');
        }

        $this->model->cancelar($id);
        AuthMiddleware::clearLicenciaCache((int) $licencia['empresa_id']);
        AuditLog::log('cancelar_licencia', 'licencias', $id, $licencia);
        flash('success', 'Licencia cancelada.');
        redirect('/licencias');
    }
}
