<?php
/**
 * OptiCore SaaS - EmpresaController (Solo SuperAdmin)
 */

class EmpresaController
{
    private Empresa $model;

    public function __construct()
    {
        Auth::requireSuperAdmin();
        $this->model = new Empresa();
    }

    // ── GET /empresas ─────────────────────────────────────────
    public function index(): void
    {
        $page     = currentPage();
        $busqueda = get('q');
        $result   = $this->model->getAll($page, $busqueda);
        view('empresas.index', [
            'title'      => 'Empresas',
            'empresas'   => $result['data'],
            'pagination' => $result,
            'busqueda'   => $busqueda,
        ]);
    }

    // ── GET /empresas/create ──────────────────────────────────
    public function create(): void
    {
        $planes = (new Plan())->getActivos();
        view('empresas.create', ['title' => 'Nueva Empresa', 'planes' => $planes]);
    }

    // ── POST /empresas ────────────────────────────────────────
    public function store(): void
    {
        csrf_verify();

        $errors = validateRequired($_POST, [
            'nombre'  => 'Nombre',
            'plan_id' => 'Plan',
            'email'   => 'Email',
        ]);

        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect('/empresas/create');
        }

        if (!filter_var($_POST['email'], FILTER_VALIDATE_EMAIL)) {
            flash('error', 'El email no es válido.');
            redirect('/empresas/create');
        }

        $db = db();
        $db->beginTransaction();
        try {
            // Crear empresa
            $empresaId = $this->model->crear($_POST);

            // Crear licencia de prueba automática
            $planId = (int) $_POST['plan_id'];
            $plan   = (new Plan())->getById($planId);
            $db->insert('licencias', [
                'empresa_id'   => $empresaId,
                'plan_id'      => $planId,
                'tipo'         => 'prueba',
                'fecha_inicio' => date('Y-m-d'),
                'fecha_fin'    => date('Y-m-d', strtotime('+30 days')),
                'monto'        => 0,
                'estado'       => 'activa',
                'observaciones'=> 'Licencia de prueba automática',
            ]);

            // Crear sucursal principal
            $db->insert('sucursales', [
                'empresa_id'  => $empresaId,
                'nombre'      => 'Casa Matriz',
                'es_principal'=> 1,
                'estado'      => 'activa',
            ]);

            $db->commit();
            AuditLog::log('crear_empresa', 'empresas', $empresaId, null, $_POST);
            flash('success', 'Empresa creada correctamente con licencia de prueba (30 días).');
            redirect('/empresas');
        } catch (Exception $e) {
            $db->rollBack();
            flash('error', 'Error al crear la empresa: ' . $e->getMessage());
            redirect('/empresas/create');
        }
    }

    // ── GET /empresas/{id} ────────────────────────────────────
    public function show(int $id): void
    {
        $empresa = $this->model->getById($id);
        if (!$empresa) {
            flash('error', 'Empresa no encontrada.');
            redirect('/empresas');
        }

        $licencias  = (new Licencia())->getByEmpresa($id);
        $sucursales = db()->fetchAll(
            "SELECT * FROM sucursales WHERE empresa_id = ? ORDER BY es_principal DESC, nombre",
            [$id]
        );
        $usuarios = db()->fetchAll(
            "SELECT u.*, r.nombre as rol_nombre FROM usuarios u
             JOIN roles r ON u.rol_id = r.id
             WHERE u.empresa_id = ? ORDER BY u.nombre",
            [$id]
        );

        view('empresas.show', [
            'title'     => $empresa['nombre'],
            'empresa'   => $empresa,
            'licencias' => $licencias,
            'sucursales'=> $sucursales,
            'usuarios'  => $usuarios,
        ]);
    }

    // ── GET /empresas/{id}/edit ───────────────────────────────
    public function edit(int $id): void
    {
        $empresa = $this->model->getById($id);
        if (!$empresa) {
            flash('error', 'Empresa no encontrada.');
            redirect('/empresas');
        }
        $planes = (new Plan())->getActivos();
        view('empresas.edit', [
            'title'   => 'Editar Empresa',
            'empresa' => $empresa,
            'planes'  => $planes,
        ]);
    }

    // ── POST /empresas/{id} ───────────────────────────────────
    public function update(int $id): void
    {
        csrf_verify();

        $empresa = $this->model->getById($id);
        if (!$empresa) {
            flash('error', 'Empresa no encontrada.');
            redirect('/empresas');
        }

        $errors = validateRequired($_POST, ['nombre' => 'Nombre', 'plan_id' => 'Plan']);
        if ($errors) {
            flash('error', implode('<br>', $errors));
            redirect("/empresas/$id/edit");
        }

        // Subir logo si se envió
        $logoPath = $empresa['logo'];
        if (!empty($_FILES['logo']['name'])) {
            $uploaded = uploadFile($_FILES['logo'], 'logos', ['jpg','jpeg','png','gif','svg']);
            if ($uploaded) $logoPath = $uploaded;
        }

        $this->model->actualizar($id, array_merge($_POST, ['logo' => $logoPath]));
        AuditLog::log('editar_empresa', 'empresas', $id, $empresa, $_POST);
        flash('success', 'Empresa actualizada correctamente.');
        redirect('/empresas');
    }

    // ── POST /empresas/{id}/delete ────────────────────────────
    public function destroy(int $id): void
    {
        csrf_verify();

        $empresa = $this->model->getById($id);
        if (!$empresa) {
            flash('error', 'Empresa no encontrada.');
            redirect('/empresas');
        }

        $this->model->eliminar($id);
        AuditLog::log('eliminar_empresa', 'empresas', $id, $empresa);
        flash('success', 'Empresa eliminada correctamente.');
        redirect('/empresas');
    }
}
